using System.Collections.Generic;
using System.Linq;
using RimWorld;
using UnityEngine;
using Verse;
using Verse.AI;

namespace Therapy
{
    internal static class MainUtility
    {
        private const float ChairRadius = 3.9f;
        private static readonly int NumRadiusCells = GenRadial.NumCellsInRadius(ChairRadius);
        private static readonly List<IntVec3> RadialPatternMiddleOutward =
            (from c in GenRadial.RadialPattern.Take(NumRadiusCells)
                orderby Mathf.Abs((c - IntVec3.Zero).LengthHorizontal - ChairRadius/2)
                select c).ToList();

        public static readonly RoomRoleDef therapyRoomRoleDef = DefDatabase<RoomRoleDef>.GetNamed("TherapyOffice");
        public static readonly JobDef therapistJobDef = DefDatabase<JobDef>.GetNamed("PerformTherapy");
        public static readonly JobDef patientJobDef = DefDatabase<JobDef>.GetNamed("ReceiveTherapy");

        private static List<IntVec3> cacheChairCells = new List<IntVec3>();

        public static List<IntVec3> ChairCellsAround(IntVec3 center, Map map, out bool hasChair)
        {
            hasChair = false;
            cacheChairCells.Clear();
            if (!center.InBounds(map)) return cacheChairCells;
            Region region = center.GetRegion(map);
            if (region == null) return cacheChairCells;

            foreach (var pos in RadialPatternMiddleOutward)
            {
                var c = pos + center;
                if (!c.InBounds(map)) continue;
                if (!c.Walkable(map)) continue;
                if (!GenSight.LineOfSight(center, c, map, true)) continue;

                Building edifice = c.GetEdifice(map);
                if (edifice != null && edifice.def.building.isSittable)
                {
                    hasChair = true;
                }
                cacheChairCells.Add(c);
            }
            return cacheChairCells;
        }


        public static Building_Couch CurrentCouch(this Pawn p)
        {
            if (!p.Spawned || p.CurJob == null || p.jobs.curDriver.layingDown != LayingDownState.LayingSurface) return null;
            Building_Couch couch = null;
            List<Thing> thingList = p.Position.GetThingList(p.Map);
            foreach (Thing thing in thingList)
            {
                couch = thing as Building_Couch;
                if (couch != null) break;
            }
            if (couch == null) return null;
            if (couch.GetCurOccupant() == p) return couch;
            return null;
        }

        public static bool TryFindChairNearCouch(Building_Couch couch, out Thing chair)
        {
            foreach (IntVec3 v in RadialPatternMiddleOutward)
            {
                IntVec3 c = couch.Position + v;
                if (!c.InBounds(couch.Map)) continue;
                if (!c.Walkable(couch.Map)) continue; 
                Building edifice = c.GetEdifice(couch.Map);
                if (edifice != null && edifice.def.building.isSittable
                    && GenSight.LineOfSight(couch.Position, edifice.Position, couch.Map, true))
                {
                    chair = edifice;
                    return true;
                }
            }
            chair = null;
            return false;
        }

        public static void FailOnCouchNoLongerUsable(this Toil toil, TargetIndex couchIndex)
        {
            toil.FailOnDespawnedOrNull(couchIndex);
            toil.FailOn(() => ((Building_Couch) toil.actor.CurJob.GetTarget(couchIndex).Thing).IsBurning());
            toil.FailOn(() => HealthAIUtility.ShouldSeekMedicalRestUrgent(toil.actor));
            toil.FailOn(() =>
                    toil.actor.IsColonist && !toil.actor.CurJob.ignoreForbidden && !toil.actor.Downed
                    && toil.actor.CurJob.GetTarget(couchIndex).Thing.IsForbidden(toil.actor));
        }

        public static float GetTherapyNeed(this Pawn pawn)
        {
            if (pawn.needs.mood == null) return 0;
            var memories = pawn.needs.mood.thoughts.memories.Memories.Where(m => m.VisibleInNeedsTab && m.MoodOffset() < 0 && m.def.durationDays > 0);
            var sum = memories.Sum(m => -m.MoodOffset());
            return sum;
        }
    }
}